/* ***************************************************** **
   ch19_merging_date_ranges.sql
   
   Skrypt dla książki Praktyczna nauka SQL dla Oracle, Helion (2022),
   napisanej przez Kima Berga Hansena, https://www.kibeha.dk
   Używasz na własną odpowiedzialność.
   *****************************************************
   
   Rozdział 19.
   Łączenie zakresów dat
   
   Skrypt przeznaczony do wykonania w schemacie PRACTICAL
** ***************************************************** */

/* -----------------------------------------------------
   Konfiguracja formatowania sqlcl
   ----------------------------------------------------- */

set pagesize 80
set linesize 80
set sqlformat ansiconsole

alter session set nls_date_format = 'YYYY-MM-DD';

/* -----------------------------------------------------
   Przykładowy kod do rozdziału 19.
   ----------------------------------------------------- */

-- Listing 19.2. Zapytanie wyświetlające okresy zatrudnienia pracowników

select
   ehp.emp_id
 , ehp.name
 , ehp.start_date
 , ehp.end_date
 , ehp.title
from emp_hire_periods_with_name ehp
order by ehp.emp_id, ehp.start_date;

-- Listing 19.4. Wykonywanie zapytania do tabeli okresów

select
   ehp.emp_id
 , e.name
 , ehp.start_date
 , ehp.end_date
 , ehp.title
from emp_hire_periods
        as of period for employed_in date '2010-07-01'
     ehp
join employees e
   on e.id = ehp.emp_id
order by ehp.emp_id, ehp.start_date;

-- Wykonanie zapytania dotyczącego innej daty

select
   ehp.emp_id
 , e.name
 , ehp.start_date
 , ehp.end_date
 , ehp.title
from emp_hire_periods
        as of period for employed_in date '2016-07-01'
     ehp
join employees e
   on e.id = ehp.emp_id
order by ehp.emp_id, ehp.start_date;

-- Listing 19.5. Porównywanie wartości start_date z wartością end_date poprzedniego rekordu

select
   emp_id
 , name
 , start_date
 , end_date
 , jobs
from emp_hire_periods_with_name
match_recognize (
   partition by emp_id
   order by start_date, end_date
   measures
      max(name)         as name
    , first(start_date) as start_date
    , last(end_date)    as end_date
    , count(*)          as jobs
   pattern (
      strt adjoin_or_overlap*
   )
   define
      adjoin_or_overlap as
         start_date <= prev(end_date)
)
order by emp_id, start_date;

-- Próba ułożenia danych w kolejności według najpierw end_date zamiast start_date

select
   emp_id
 , name
 , start_date
 , end_date
 , jobs
from emp_hire_periods_with_name
match_recognize (
   partition by emp_id
   order by end_date, start_date
   measures
      max(name)         as name
    , first(start_date) as start_date
    , last(end_date)    as end_date
    , count(*)          as jobs
   pattern (
      strt adjoin_or_overlap*
   )
   define
      adjoin_or_overlap as
         start_date <= prev(end_date)
)
order by emp_id, start_date;

-- Próba porównania start_date z dotychczas największą wartością end_date
-- To nie działa i w zależności od wersji DB i klienta ryzykujesz,
-- że sesja zakończy się awarią i zgłoszeniem jednego z następujących błędów:
--    ORA-03113: end-of-file on communication channel
--    java.lang.NullPointerException
-- Nie używaj tego polecenia, ma ono jedynie charakter dydaktyczny
/*
select
   emp_id
 , name
 , start_date
 , end_date
 , jobs
from emp_hire_periods_with_name
match_recognize (
   partition by emp_id
   order by start_date, end_date
   measures
      max(name)         as name
    , first(start_date) as start_date
    , max(end_date)     as end_date
    , count(*)          as jobs
   pattern (
      strt adjoin_or_overlap*
   )
   define
      adjoin_or_overlap as
         start_date <= max(end_date)
)
order by emp_id, start_date;
*/

-- Listing 19.6. Porównywanie wartości start_date następnego rekordu z dotychczas największą wartością end_date

select
   emp_id
 , name
 , start_date
 , end_date
 , jobs
from emp_hire_periods_with_name
match_recognize (
   partition by emp_id
   order by start_date, end_date
   measures
      max(name)         as name
    , first(start_date) as start_date
    , max(end_date)     as end_date
    , count(*)          as jobs
   pattern (
      adjoin_or_overlap* last_row
   )
   define
      adjoin_or_overlap as
         next(start_date) <= max(end_date)
)
order by emp_id, start_date;

-- Listing 19.7. Obsługa wartości null dla dat początkowej i końcowej okresu zatrudnienia

select
   emp_id
 , name
 , start_date
 , end_date
 , jobs
from emp_hire_periods_with_name
match_recognize (
   partition by emp_id
   order by start_date nulls first, end_date nulls last
   measures
      max(name)         as name
    , first(start_date) as start_date
    , nullif(
         max(nvl(end_date, date '9999-12-31'))
       , date '9999-12-31'
      )                 as end_date
    , count(*)          as jobs
   pattern (
      adjoin_or_overlap* last_row
   )
   define
      adjoin_or_overlap as
         nvl(next(start_date), date '-4712-01-01')
            <= max(nvl(end_date, date '9999-12-31'))
)
order by emp_id, start_date;

/* ***************************************************** */
